/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved.
*/

#ifndef UUID_a4039c97_9ae4_479a_bedb_29ed91fb1329
#define UUID_a4039c97_9ae4_479a_bedb_29ed91fb1329

/**
 * \file vnccdb.h
 *
 * This file contains prototypes for those VNC CDB and SBP SDK APIs that act on
 * CDB endpoints.
 *
 * The application should normally include vnccdbsdk.h rather than including
 * this file directly.
 *
 * \see vnccdbsdk.h, vnccdbtypes.h
 */

#include "vnccall.h"
#include "vncint.h"
#include "vnccdbtypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Tests whether the given service ID is legal as the ID of a
 * client-side CDB service.
 *
 * \warning VNCCDBServiceIsClientSide() is a preprocessor macro. Its arguments
 * may be evaluated more than once.
 *
 * \return Non-zero if the service ID is legal as the ID of a client-side CDB
 * service, or zero if not.
 */
#define VNCCDBServiceIsClientSide(serviceId) \
    ((serviceId) >= 0x0001 && (serviceId) <= 0x3fff)

/**
 * \brief Tests whether the given service ID is legal as the ID of a
 * server-side CDB service.
 *
 * \warning VNCCDBServiceIsClientSide() is a preprocessor macro. Its arguments
 * may be evaluated more than once.
 *
 * \return Non-zero if the service ID is legal as the ID of a server-side CDB
 * service, or zero if not.
 */
#define VNCCDBServiceIsServerSide(serviceId) \
    ((serviceId) >= 0x4001 && (serviceId) <= 0x7fff)

/** \defgroup cdb CDB functions
 * @{
 */

/**
 * \brief Called by the SDK when a source CDB endpoint receives a StartService
 * message from its peer CDB sink.
 *
 * The application should perform any action necessary to start the service and
 * then call VNCCDBEndpointSendServiceResponse() to inform the peer endpoint of
 * the outcome. This call may be made either from within the
 * VNCCDBEndpointStartServiceCallback() or at a later time. If the service is
 * started successfully, then the response should be
 * ::VNCCDBServiceResponseOKServiceStarted.
 *
 * The SDK does not track whether or not CDB services are running. It is the
 * application's responsibility to do this, and to respond with
 * ::VNCCDBServiceResponseWarningServiceRunning instead of
 * ::VNCCDBServiceResponseOKServiceStarted if the service is already running.
 *
 * However the SDK does track the state of SBP services, and will issue
 * suitable ServiceResponse messages automatically. It is therefore not
 * normally necessary for applications to respond for an SBP service.
 *
 * \param pEndpoint The CDB endpoint that is the source for this service.
 * \param pEndpointContext Endpoint context pointer specified by the
 * application in the ::VNCCDBEndpointDescriptor.
 * \param serviceId The ID of this service, as set by
 * VNCCDBEndpointAddCDBSource().
 * \param pServiceContext Service context pointer specified by the
 * application in the ::VNCCDBService structure.
 * \param versionMajor Service version requested by the peer endpoint.
 * \param versionMinor Service version requested by the peer endpoint.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointStartServiceCallback(VNCCDBEndpoint *pEndpoint,
                                   void *pEndpointContext,
                                   VNCCDBServiceId serviceId,
                                   void *pServiceContext,
                                   vnc_uint8_t versionMajor,
                                   vnc_uint8_t versionMinor);

/**
 * \brief Called by the SDK when a source CDB endpoint receives a StopService
 * message from its peer CDB sink.
 *
 * The application should perform any action necessary to stop the service and
 * then call VNCCDBEndpointSendServiceResponse() to inform the peer endpoint of
 * the outcome. This call may be made either from within the
 * VNCCDBEndpointStopServiceCallback() or at a later time. If the service is
 * started successfully, then the response should be
 * ::VNCCDBServiceResponseOKServiceStopped.
 *
 * The SDK does not track whether or not CDB services are running. It is the
 * application's responsibility to do this, and to respond with
 * ::VNCCDBServiceResponseWarningServiceNotRunning instead of
 * ::VNCCDBServiceResponseOKServiceStopped if the service is not currently
 * running.
 *
 * However the SDK does track the state of SBP services, and will issue
 * suitable ServiceResponse messages automatically. It is not normally
 * necessary for applications to respond for an SBP service.
 *
 * \param pEndpoint The CDB endpoint that is the source for this service.
 * \param pEndpointContext Endpoint context pointer specified by the
 * application in the ::VNCCDBEndpointDescriptor.
 * \param serviceId The ID of this service, as set by
 * VNCCDBEndpointAddCDBSource().
 * \param pServiceContext Service context pointer specified by the
 * application in the ::VNCCDBService structure.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointStopServiceCallback(VNCCDBEndpoint *pEndpoint,
                                  void *pEndpointContext,
                                  VNCCDBServiceId serviceId,
                                  void *pServiceContext);

/**
 * \brief Called by the SDK when a CDB endpoint receives a ServicePayload
 * message.
 *
 * Callbacks of this type are used both by CDB sources and CDB sinks. In the
 * case of a CDB source, the serviceId parameter identifies the service to
 * which the payload has been sent; in the case of a CDB sink, the serviceId
 * parameter identifies the service that sent the payload.
 *
 * The SDK filters received ServicePayload messages and discards any that do
 * not contain valid service IDs or that otherwise violate the CDB protocol.
 * ServiceResponse messages are generated for such error cases automatically.
 *
 * However, once the SDK has passed the payload to the application, it becomes
 * the application's responsibility to detect further error conditions. In
 * particular, the SDK does not track whether or not CDB services are running.
 * It is therefore the application's responsibility to respond with
 * ::VNCCDBServiceResponseWarningServiceNotRunning if necessary.
 *
 * The SDK will handle payloads for SBP service internally, and will issue
 * suitable ServiceResponse messages automatically. It is not normally
 * necessary for applications to respond for an SBP service.
 *
 * \param pEndpoint The CDB endpoint that is the source for this service.
 * \param pEndpointContext Endpoint context pointer specified by the
 * application in the ::VNCCDBEndpointDescriptor.
 * \param serviceId The ID of this service, as set by
 * VNCCDBEndpointAddCDBSource().
 * \param pServiceContext Service context pointer specified by the
 * application in the ::VNCCDBService structure.
 * \param payload The payload data from the ServicePayload message. This
 * pointer is valid only until the callback returns.
 * \param payloadSize The length of the data to which the payload parameter
 * points.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointServicePayloadCallback(VNCCDBEndpoint *pEndpoint,
                                     void *pEndpointContext,
                                     VNCCDBServiceId serviceId,
                                     void *pServiceContext,
                                     const vnc_uint8_t *payload,
                                     size_t payloadLength);

/**
 * \brief Called by the SDK when a sink CDB endpoint receives a ServiceResponse
 * message from its peer CDB source.
 *
 * The application should use this notification to determine the outcome of the
 * StartService, StopService and ServicePayload messages that it sends. It is
 * recommended that the application call VNCCDBEndpointSendStartService() and then
 * wait for a ::VNCCDBServiceResponseWarningServiceRunning or
 * ::VNCCDBServiceResponseOKServiceStarted ServiceResponse before sending a
 * ServicePayload to a peer CDB service.
 *
 * Note that, as per \ref ml_cdb "[ML-CDB]", there is no ServiceResponse
 * message in response to ServicePayload messages that are received and
 * processed successfully.
 *
 * In cases where \ref ml_cdb "[ML-CDB]" requires that a particular
 * ServiceResponse causes the CDB session to be terminated, the SDK will
 * terminate the session automatically.
 *
 * In cases where no response to a StartService or StopService is received in
 * the defined timeout, this callback will be called with either
 * VNCCDBServiceResponseErrorServiceLaunchFailed or
 * VNCCDBServiceResponseErrorServiceTerminationFailed as appropriate. The SDK
 * will handle sending a StopService in response to a timed-out StartService
 * request as per the CDB specification, and applications must be written to
 * cope with the fact that this may produce an additional ServiceResponse
 * message.
 *
 * \param pEndpoint The CDB endpoint that is the source for this service.
 * \param pEndpointContext Endpoint context pointer specified by the
 * application in the ::VNCCDBEndpointDescriptor.
 * \param serviceId The ID of this service, as set by
 * VNCCDBEndpointAddCDBSource().
 * \param pServiceContext Service context pointer specified by the
 * application in the ::VNCCDBService structure.
 * \param serviceResponse ServiceResponse response code.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointServiceResponseCallback(VNCCDBEndpoint *pEndpoint,
                                      void *pEndpointContext,
                                      VNCCDBServiceId serviceId,
                                      void *pServiceContext,
                                      VNCCDBServiceResponse response);

/**
 * \brief Callbacks from the SDK to an application as a result of messages
 * received for one of this endpoint's service sources.
 */
typedef struct
{
    /** Called when a StartService message is received for this service. */
    VNCCDBEndpointStartServiceCallback *pStartCallback;
    /** Called when a StopService message is received for this service. */
    VNCCDBEndpointStopServiceCallback *pStopCallback;
    /** Called when a ServicePayload message is received for this service. */
    VNCCDBEndpointServicePayloadCallback *pPayloadCallback;
    /** Called when a ServiceResponse message is received from the service. */
    VNCCDBEndpointServiceResponseCallback *pResponseCallback;
} VNCCDBSourceCallbacks;

/**
 * \brief Callbacks from the SDK to an application as a result of messages
 * received for one of this endpoint's service sinks.
 */
typedef struct
{
    /** Called when a ServicePayload message is received from the service. */
    VNCCDBEndpointServicePayloadCallback *pPayloadCallback;
    /** Called when a ServiceResponse message is received from the service. */
    VNCCDBEndpointServiceResponseCallback *pResponseCallback;
} VNCCDBSinkCallbacks;

/**
 * \brief Notifies the application that a CDB endpoint started a new thread. 
 *
 * Callbacks made by the CDB endpoint will happen only on threads that were 
 * reported through this callback.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 */
typedef void VNCCALL
VNCCDBEndpointThreadStartedCallback(VNCCDBEndpoint *pEndpoint,
                                    void *pEndpointContext);

/**
 * \brief Notifies the application that a CDB endpoint thread is about to
 * terminate.
 *
 * No further callbacks will happen from this thread.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 * \param error The final outcome error of the thread.
 */
typedef void VNCCALL
VNCCDBEndpointThreadStoppedCallback(VNCCDBEndpoint *pEndpoint,
                                    void *pEndpointContext,
                                    VNCCDBError error);

/**
 * \brief Callback from the SDK to emit a log message associated with a CDB
 * endpoint.
 *
 * It is recommended that the application implements this callback to aid with
 * the diagnosis of any issues that may arise.
 *
 * The SDK may invoke VNCCDBEndpointLogCallback() from any thread calls SDK
 * API, as well as from the CDB thread. However, the SDK guarantees that it
 * will never enter VNCCDBEndpointLogCallback() from more than one thread at
 * a time.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 * \param category Log message category, as a NUL-terminated UTF-8 string.
 * \param severity Log message severity. Severities range from 0 (error) 
 * to 100 (debug).
 * \param text Log message text, as a NUL-terminated UTF-8 string.
 *
 * \see VNCCDBEndpointCreate(), ::VNCCDBEndpointCallbacks
 */
typedef void VNCCALL
VNCCDBEndpointLogCallback(VNCCDBEndpoint *pEndpoint,
                          void *pContext,
                          const char *category,
                          vnc_int32_t severity,
                          const char *text);

/**
 * \brief Callback from the SDK when the status of a CDB endpoint changes.
 *
 * This callback is made at several points during the lifetime of a CDB
 * endpoint's CDB thread. The status parameter indicates the current status of
 * the CDB thread.
 *
 * The application can use this callback to determine when the CDB session is
 * established or lost, and when (and why) the CDB thread is exiting. See
 * ::VNCCDBEndpointStatus for further details.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 * \param status The endpoint's current status.
 * \param additionalInformation For certain values of status, this parameter
 * may provide additional information. See ::VNCCDBEndpointStatus for further
 * details.
 *
 * \see VNCCDBEndpointCreate(), ::VNCCDBEndpointCallbacks,
 * ::VNCCDBEndpointStatus
 */
typedef void VNCCALL
VNCCDBEndpointStatusCallback(VNCCDBEndpoint *pEndpoint,
                             void *pEndpointContext,
                             VNCCDBEndpointStatus status,
                             vnc_intptr_t additionalInformation);

/**
 * \brief Callback from the SDK when new services are seen in a
 * ServicesSupported message from the peer endpoint.
 *
 * This callback is invoked by the SDK for each ServicesSupported message that
 * is received from the peer endpoint where new services are seen (i.e. those
 * which haven't been seen in a previous ServicesSupported message).
 *
 * Services are only provided to the application once via this callback so long
 * as the service ID, name and major version remain unchanged. If any of these
 * values differ, then the service is taken as a different one.
 *
 * If a service indicated in this callback is missing from a later
 * ServicesSupported message (indicating that the source no longer offers 
 * the service) then the application is alerted to this via
 * VNCCDBEndpointServicesNotSupportedCallback() and any sink registered for
 * those services are automatically unregistered.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 * \param versionMajor Major component of the CDB protcol version supported by
 * the peer endpoint.
 * \param versionMinor Major component of the CDB protcol version supported by
 * the peer endpoint.
 * \param services The array describing the services supported by the peer
 * endpoint.
 * \param serviceSize The size of each ::VNCCDBService structure in the
 * services array.
 * \param serviceCount The number of ::VNCCDBService elements in the services
 * array.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */

typedef VNCCDBError VNCCALL
VNCCDBEndpointServicesSupportedCallback(VNCCDBEndpoint *pEndpoint,
                                        void *pEndpointContext,
                                        vnc_uint8_t versionMajor,
                                        vnc_uint8_t versionMinor,
                                        const VNCCDBService *services,
                                        size_t serviceSize,
                                        size_t serviceCount);
/**
 * \brief Callback from the SDK when previously supported services are missing
 * from a ServicesSupported message from the peer endpoint.
 *
 * This callback is invoked by the SDK for each ServicesSupported message that
 * is received from the peer endpoint where previously-offered services (that
 * were passed via VNCCDBEndpointServicesSupportedCallback()) are missing, and
 * hence no longer offered.
 *
 * Any sink registered for any of the services is automatically unregistered
 * prior to this callback being called.
 *
 * If a service passed to this callback is later re-added by the source and
 * appears in a later ServicesSupported message, it will be indicated to the
 * application via VNCCDBEndpointServicesSupportedCallback().
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 * \param versionMajor Major component of the CDB protcol version supported by
 * the peer endpoint.
 * \param versionMinor Major component of the CDB protcol version supported by
 * the peer endpoint.
 * \param services The array describing the services supported by the peer
 * endpoint.
 * \param serviceSize The size of each ::VNCCDBService structure in the
 * services array.
 * \param serviceCount The number of ::VNCCDBService elements in the services
 * array.
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointServicesNotSupportedCallback(VNCCDBEndpoint *pEndpoint,
                                           void *pEndpointContext,
                                           vnc_uint8_t versionMajor,
                                           vnc_uint8_t versionMinor,
                                           const VNCCDBService *services,
                                           size_t serviceSize,
                                           size_t serviceCount);

/**
 * \brief Callback from the SDK when a ServicesRequest message has timed out.
 *
 * This callback is invoked by the SDK for each ServicesSupported message that
 * is received from the peer endpoint where previously-offered services (that
 * were passed via VNCCDBEndpointServicesSupportedCallback()) are missing, and
 * hence no longer offered.
 *
 * Any sink registered for any of the services is automatically unregistered
 * prior to this callback being called.
 *
 * If a service passed to this callback is later re-added by the source and
 * appears in a later ServicesSupported message, it will be indicated to the
 * application via VNCCDBEndpointServicesSupportedCallback().
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext The endpoint context, as supplied in the
 * ::VNCCDBEndpointDescriptor in the call to VNCCDBEndpointCreate().
 *
 * \return The SDK does not examine the return value from this callback.
 * However, for future compatibility, the application should return
 * ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointServicesRequestTimeout(VNCCDBEndpoint *pEndpoint,
                                     void *pEndpointContext);

/**
 * \brief Callbacks from the SDK to the application as the results of events
 * that do not originate with a particular service.
 *
 * \see VNCCDBEndpointCreate()
 */
typedef struct
{
    /** Notifies the application that a CDB endpoint started a new thread. */ 
    VNCCDBEndpointThreadStartedCallback *pThreadStartedCallback;
    
    /** Notifies the application that a CDB endpoint thread is about to terminate. */ 
    VNCCDBEndpointThreadStoppedCallback *pThreadStoppedCallback;
  
    /** Informs the application of a log message emitted by a CDB endpoint. */
    VNCCDBEndpointLogCallback *pLogCallback;

    /** Informs the application of a change in the status of a CDB endpoint. */
    VNCCDBEndpointStatusCallback *pStatusCallback;

    /**
     * Informs the application of newly supported services after a
     * ServicesSupported message received from the * peer CDB endpoint.
     */
    VNCCDBEndpointServicesSupportedCallback *pServicesSupportedCallback;

    /**
     * Informs the application of services no longer supported after a
     * ServicesSupported message received from the peer CDB endpoint.
     */
    VNCCDBEndpointServicesNotSupportedCallback *pServicesNotSupportedCallback;

    /** Informs the application that the ServicesRequest has timed out. */
    VNCCDBEndpointServicesRequestTimeout *pServicesTimeoutCallback;
} VNCCDBEndpointCallbacks;

/**
 * \brief Used with VNCCDBEndpointCreate() to define a CDB endpoint.
 *
 * \see VNCCDBEndpointCreate(), ::VNCCDBEndpointType
 */
typedef struct
{
    /**
     * \brief CDB URL. Should be of the form CDB://A.B.C.D:P, where A.B.C.D is
     * an IPv4 address in dotted-quad representation, and P is a TCP port
     * number in decimal.
     *
     * If this is a client endpoint, then, when the endpoint is started, an
     * outgoing TCP connection is made to A.B.C.D:P.
     *
     * If this is a server endpoint, then, when the endpoint is started, the
     * SDK listens on A.B.C.D:P for a single incoming TCP connection.
     */
    const char *url;

    /**
     * \brief Denotes whether this is a client endpoint or a server endpoint.
     */
    VNCCDBEndpointType endpointType;

    /**
     * \brief Opaque data pointer that the application may use to associate its
     * own data with the CDB endpoint.
     *
     * The SDK does not interact in any with the data pointed to. In
     * particular, the SDK does not free the data pointed to when the endpoint
     * is destroyed.
     */
    void *pContext;

    /**
     * Major component of the CDB protocol version supported by this endpoint.
     */
    vnc_uint8_t versionMajor;

    /**
     * Minor component of the CDB protocol version supported by this endpoint.
     */
    vnc_uint8_t versionMinor;
} VNCCDBEndpointDescriptor;

/**
 * \brief Creates a CDB endpoint.
 *
 * CDB endpoints created by VNCCDBEndpointCreate() may be used as CDB sources,
 * CDB sinks, or both. For further details on the usage of CDB endpoints, see
 * ::VNCCDBEndpoint.
 *
 * The application should call VNCCDBEndpointDestroy() to destroy the CDB
 * endpoint when it is no longer required.
 *
 * \param ppEndpoint On successful return, *ppEndpoint points to the new CDB
 * endpoint.
 * \param pDescriptor Pointer to a structure containing details about the CDB
 * endpoint.
 * \param descriptorSize Size of the structure pointed to be *pDescriptor.
 * \param pCallbacks Pointer to a structure containing the addresses of
 * callback functions associated with the endpoint.
 * \param callbacksSize Size of the structure pointed to by *pCallbacks.
 *
 * \retval ::VNCCDBErrorNone The endpoint was created successfully.
 * \retval ::VNCCDBErrorInvalidParameter One or more of the parameters is
 * invalid.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 *
 * \see ::VNCCDBEndpoint, VNCCDBEndpointDestroy(), ::VNCCDBEndpointDescriptor,
 * ::VNCCDBEndpointCallbacks
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointCreate(VNCCDBEndpoint **ppEndpoint,
                     const VNCCDBEndpointDescriptor *pDescriptor,
                     size_t descriptorSize,
                     const VNCCDBEndpointCallbacks *pCallbacks,
                     size_t callbacksSize);

/**
 * \brief Destroys a CDB endpoint.
 *
 * If the endpoint is running, then it is stopped first. As with
 * VNCCDBEndpointStop(), VNCCDBEndpointDestroy() does not return until the CDB
 * thread has exited.
 *
 * \param pEndpoint The CDB endpoint.
 *
 * \retval ::VNCCDBErrorNone The endpoint was destroyed successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointDestroy(VNCCDBEndpoint *pEndpoint);

/**
 * \brief Returns the context pointer associated with a CDB endpoint.
 *
 * \param pEndpoint The CDB endpoint.
 *
 * \return The context pointer that was supplied in the
 * ::VNCCDBEndpointDescriptor when VNCCDBEndpointCreate() was called.
 *
 * \see VNCCDBEndpointDescriptor, VNCCDBEndpointCreate()
 */
typedef void *VNCCALL
VNCCDBEndpointGetContext(VNCCDBEndpoint *pEndpoint);

/**
 * \brief Adds a license to a CDB endpoint.
 *
 * CDB endpoints must always be licensed before they can be used. It is
 * recommended that VNCCDBEndpointAddLicense() is called soon after each call
 * to VNCCDBEndpointCreate().
 *
 * To obtain a license for use with VNC CDB SDK, contact your RealVNC sales
 * representative.
 *
 * \param pEndpoint The CDB endpoint.
 * \param licenseText The text of the license, as a NUL-terminated ASCII
 * string.
 * \param serialNumber On successful return, if serialNumber is not NULL, then
 * it will contain the serial number of the added license. The serial number is
 * a big-endian UUID.
 * \param serialNumberSize If serialNumber is not NULL, then serialNumberSize
 * should be the size of the buffer pointed to by serialNumber, which should be
 * at least 16 bytes. If serialNumber is NULL, then serialNumberSize is
 * ignored.
 *
 * \retval ::VNCCDBErrorNone The license was added successfully.
 * \retval ::VNCCDBErrorLicenseNotValid licenseText is not a valid VNC license.
 * \retval ::VNCCDBErrorInvalidParameter Either licenseText is NULL, or
 * serialNumber is not NULL and serialNumberSize is less than 16.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointAddLicense(VNCCDBEndpoint *pEndpoint,
                         const char *licenseText,
                         vnc_uint8_t *serialNumber,
                         size_t serialNumberSize);

/**
 * \brief Adds a source service to a CDB endpoint.
 *
 * This method turns a CDB endpoint into a CDB source by adding a service to
 * it. The service is implemented by a set of callbacks that the application
 * supplies when calling this method, and which are called when the SDK
 * receives StartService, StopService or ServicePayload messages for this
 * service.
 *
 * If the service is an SBP service, then it is recommended that the
 * application use the higher-level VNCCDBEndpointAddSBPSource() in preference
 * to VNCCDBEndpointAddCDBSource().
 *
 * All calls to VNCCDBEndpointAddCDBSource() must be made before the endpoint
 * is started. Service IDs are always allocated automatically by the SDK and
 * returned in pService->serviceId.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext Opaque data pointer that the application may use to
 * associate its own data with the service.
 * \param pService Pointer to a structure describing the service. On successful
 * return, pService->serviceId is the SDK-allocated service ID for this
 * service. (The original value of pService->serviceId is ignored.)
 * \param serviceSize Size of the structure pointed to by pService.
 * \param pSourceCallbacks Pointer to a structure providing the callbacks that
 * implement the service.
 * \param sourceCallbacksSize Size of the structure pointed to by
 * pSourceCallbacks.
 *
 * \retval ::VNCCDBErrorNone The service was added successfully.
 * \retval ::VNCCDBErrorInvalidParameter One or more of the parameters is
 * invalid.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileRunning The CDB thread is running. Calls to
 * this method must be made before the endpoint is started.
 *
 * \see VNCCDBEndpointAddSBPSource(), ::VNCCDBService, ::VNCCDBSourceCallbacks
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointAddCDBSource(VNCCDBEndpoint *pEndpoint,
                           void *pContext,
                           VNCCDBService *pService,
                           size_t serviceSize,
                           const VNCCDBSourceCallbacks *pSourceCallbacks,
                           size_t sourceCallbacksSize);

/**
 * \brief Adds a sink for a CDB service hosted by the peer endpoint.
 *
 * This method adds a sink to a CDB endpoint for a CDB service hosted by the
 * peer endpoint. The sink is implemented by a set of callbacks that the
 * application supplies when calling this method, and which are called when the
 * SDK receives ServicePayload or ServiceResponse messages from the service.
 *
 * If the service is an SBP service, then it is recommended that the
 * application use the higher-level VNCCDBEndpointAddSBPSink() in preference
 * to VNCCDBEndpointAddCDBSink().
 *
 * Because service IDs and service availability may change from session to
 * session, and therefore cannot be known in advance, all calls to
 * VNCCDBEndpointAddCDBSink() should be made during or after the
 * VNCCDBEndpointServicesSupportedCallback(). Once registered, the sink remains
 * as long as the service ID, name and major version of the service remain
 * consistent in further ServicesSupported messages. However, if any of these
 * properties are changed by the source then the
 * VNCCDBEndpointServicesNotSupportedCallback() will be invoked and the sink
 * will need to be re-registered.
 *
 * \param pEndpoint The CDB endpoint.
 * \param pContext Opaque data pointer that the application may use to
 * associate its own data with the sink.
 * \param serviceId The service ID for which the sink will be registered.
 * \param pSinkCallbacks Pointer to a structure providing the callbacks that
 * will form the sink implementation.
 * \param sinkCallbacksSize Size of the structure pointed to by
 * pSinkCallbacks.
 *
 * \retval ::VNCCDBErrorNone The service was added successfully.
 * \retval ::VNCCDBErrorInvalidParameter One or more of the parameters is
 * invalid.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running. The
 * application should wait until receiving a ServicesSupported message before
 * registering a CDB sink.
 *
 * \see VNCCDBEndpointAddSBPSink(), ::VNCCDBService, ::VNCCDBSinkCallbacks
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointAddCDBSink(VNCCDBEndpoint *pEndpoint,
                         void *pContext,
                         VNCCDBServiceId serviceId,
                         const VNCCDBSinkCallbacks *pSinkCallbacks,
                         size_t sinkCallbacksSize);

/**
 * \brief Starts a CDB endpoint.
 *
 * This method starts the CDB thread for the given CDB endpoint. Once started,
 * the CDB thread attempts to establish the TCP connection to the peer CDB
 * endpoint. If this is successful, then a ServicesRequest message is
 * automatically sent.
 *
 * \param pEndpoint The endpoint.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileRunning The CDB thread is already running.
 *
 * \see VNCCDBEndpointStop()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointStart(VNCCDBEndpoint *pEndpoint);

/**
 * \brief Stops a CDB endpoint.
 *
 * This method stops the CDB thread for the given CDB endpoint. If the session
 * has been established, then it is terminated in accordance with section
 * 2.2.1.2. of \ref ml_cdb "[ML-CDB]".
 *
 * This method does not return until the CDB session has been terminated and
 * the CDB thread has exited. Be aware that, if the peer endpoint does not
 * respond to the ByeBye message, then this can take up to five seconds.
 *
 * \param pEndpoint The endpoint.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running.
 *
 * \see VNCCDBEndpointStart()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointStop(VNCCDBEndpoint *pEndpoint);

/**
 * \brief Sends a StartService message to the peer CDB endpoint.
 *
 * This method instructs the peer CDB endpoint to start a particular service.
 * Typically, the application will use this method in conjunction with
 * VNCCDBEndpointAddSBPSink() or VNCCDBEndpointAddCDBSink(), and then wait for
 * the source endpoint's ServiceResponse before interacting with the service
 * further. The source endpoint's ServiceResponse will be notified by a call to
 * the sink's VNCCDBEndpointServiceResponseCallback().
 *
 * The SDK does not track whether or not CDB services are running. It is the
 * application's responsibility to do ensure that CDB services hosted by the
 * peer have been started before attempting to communicate with them. The
 * application should check that the ServiceResponse is either
 * ::VNCCDBServiceResponseOKServiceStarted or
 * ::VNCCDBServiceResponseWarningServiceRunning before interacting with the
 * service further.
 *
 * \param pEndpoint The endpoint.
 * \param serviceId The service ID of the service that should be started. This
 * must be one of the services advertised the last ServicesSupported message
 * received by from the peer endpoint.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running.
 *
 * \see VNCCDBEndpointSendStopService(), VNCCDBEndpointAddSBPSink(),
 * VNCCDBEndpointAddCDBSink()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointSendStartService(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId serviceId);

/**
 * \brief Sends a StopService message to the peer CDB endpoint.
 *
 * This method instructs the peer CDB endpoint to stop a particular service.
 * The source endpoint will respond with a ServiceResponse, which will be
 * notified by a call to the sink's VNCCDBEndpointServiceResponseCallback().
 *
 * \param pEndpoint The endpoint.
 * \param serviceId The service ID of the service that should be started. This
 * must be one of the services advertised the last ServicesSupported message
 * received by from the peer endpoint.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running.
 *
 * \see VNCCDBEndpointSendStartService()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointSendStopService(VNCCDBEndpoint *pEndpoint,
                              VNCCDBServiceId serviceId);

/**
 * \brief Sends a CDB ServicePayload message to the peer CDB endpoint.
 *
 * This method is for use by both source endpoints and sink endpoints. In the
 * case of a sink endpoint, the serviceId parameter identifies the service
 * hosted by the peer endpoint to which the payload is addressed. However, in
 * the case of a source endpoint, the serviceId parameter identifies the
 * service hosted by this endpoint that has generated the payload.
 *
 * Sink endpoints should ensure that the peer endpoint has started a service
 * before sending any ServicePayloads to it. Use
 * VNCCDBEndpointSendStartService() to request that the peer endpoint starts a
 * service, and wait for the corresponding
 * VNCCDBEndpointServiceResponseCallback() to be informed of the outcome.
 *
 * \param pEndpoint The endpoint.
 * \param serviceId The service ID (see explanation above).
 * \param payload Pointer to the payload data.
 * \param payloadLength Length of the payload data. The maximum length allowed
 * by the protocol is 8100 bytes.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL, or payload is NULL
 * and payloadLength is non-zero, or payloadLength is greater than the maximum
 * of 8100 bytes.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running.
 *
 * \see VNCCDBEndpointSendStartService(), VNCCDBEndpointSendServiceResponse()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointSendServicePayload(VNCCDBEndpoint *pEndpoint,
                                 VNCCDBServiceId serviceId,
                                 const vnc_uint8_t *payload,
                                 size_t payloadLength);

/**
 * \brief Sends a CDB ServiceResponse message.
 *
 * Source endpoints must use this API to inform the peer endpoint of the
 * outcome of a StartService or StopService request. The call to
 * VNCCDBEndpointSendServiceResponse() may be made either from within the
 * VNCCDBEndpointStartServiceCallback(), or the
 * VNCCDBEndpointStopServiceCallback(), or at a later time.
 *
 * Both source and sink endpoints may also use this API to inform the peer
 * of errors, e.g. ::VNCCDBServiceResponseErrorUnsupportedPayloadFormat.
 *
 * \param pEndpoint The endpoint.
 * \param serviceId The ID of the service that is sending the response.
 * \param response The response code.
 *
 * \retval ::VNCCDBErrorNone The service was started successfully.
 * \retval ::VNCCDBErrorInvalidParameter pEndpoint is NULL.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 * \retval ::VNCCDBErrorIllegalWhileNotRunning The CDB thread is not running.
 *
 * \see VNCCDBEndpointSendStartService(), VNCCDBEndpointSendServicePayload()
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointSendServiceResponse(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId serviceId,
                                  VNCCDBServiceResponse response);


/** @} */


#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_a4039c97_9ae4_479a_bedb_29ed91fb1329) */
